'use client';

import React, { useState, useEffect } from 'react';
import { 
  ArrowLeft,
  Edit,
  Users,
  UserPlus,
  Settings,
  Activity,
  TrendingUp,
  Calendar,
  Building2,
  Crown,
  Shield,
  UserCheck,
  Clock
} from 'lucide-react';
import Link from 'next/link';
import { useParams } from 'next/navigation';
import DataTable, { type TableColumn, type TableAction } from '@/components/ui/DataTable';
import type { Department, AgencyUser } from '@/types';
import { departmentApi, agencyUserApi } from '@/services/api';
import { PERMISSION_CATEGORIES } from '@/utils/permissions';

export default function DepartmentDetailPage() {
  const params = useParams();
  const departmentId = params.id as string;
  
  const [department, setDepartment] = useState<Department | null>(null);
  const [users, setUsers] = useState<AgencyUser[]>([]);
  const [loading, setLoading] = useState(true);
  const [activeTab, setActiveTab] = useState<'overview' | 'users' | 'settings'>('overview');

  useEffect(() => {
    if (departmentId) {
      fetchDepartment();
      fetchDepartmentUsers();
    }
  }, [departmentId]);

  const fetchDepartment = async () => {
    try {
      setLoading(true);

      // Fetch department from API
      const data = await departmentApi.getDepartment(departmentId);

      // Convert API response to frontend format
      const deptWithDates: Department = {
        id: data.id,
        name: data.name,
        description: data.description || '',
        color: data.color,
        parentDepartmentId: data.parent_department_id,
        managerId: data.manager_id,
        permissions: data.permissions ? (typeof data.permissions === 'string' ? JSON.parse(data.permissions) : data.permissions) : [],
        isActive: Boolean(data.is_active),
        userCount: data.user_count || 0,
        createdAt: new Date(data.created_at),
        updatedAt: new Date(data.updated_at)
      };

      // Fetch manager if exists
      if (data.manager_id) {
        try {
          const managerData = await agencyUserApi.getAgencyUser(data.manager_id);
          deptWithDates.manager = {
            id: managerData.id,
            firstName: managerData.first_name,
            lastName: managerData.last_name,
            email: managerData.email,
            phone: managerData.phone,
            avatar: managerData.avatar,
            departmentId: managerData.department_id,
            position: managerData.position,
            role: managerData.role,
            permissions: managerData.permissions ? JSON.parse(managerData.permissions) : [],
            isActive: Boolean(managerData.is_active),
            isOnline: Boolean(managerData.is_online),
            lastLoginAt: managerData.last_login_at ? new Date(managerData.last_login_at) : undefined,
            createdAt: new Date(managerData.created_at),
            updatedAt: new Date(managerData.updated_at),
            department: deptWithDates
          };
        } catch (error) {
          console.error('Error fetching manager:', error);
        }
      }

      setDepartment(deptWithDates);
    } catch (error) {
      console.error('Error fetching department:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchDepartmentUsers = async () => {
    try {
      // Fetch users in this department from API
      const usersData = await agencyUserApi.getAgencyUsers({ department_id: departmentId });

      // Convert users data
      const usersWithDates = usersData.map((user: {
        id: string;
        first_name: string;
        last_name: string;
        email: string;
        phone?: string;
        avatar?: string;
        department_id: string;
        department_name?: string;
        department_color?: string;
        position?: string;
        role: string;
        permissions?: string | string[];
        is_active: boolean | number;
        is_online?: boolean | number;
        last_login_at?: string;
        total_tickets?: number;
        total_revenue?: number;
        created_at: string;
        updated_at: string;
      }) => ({
        id: user.id,
        firstName: user.first_name,
        lastName: user.last_name,
        email: user.email,
        phone: user.phone,
        avatar: user.avatar,
        departmentId: user.department_id,
        department: {
          id: user.department_id,
          name: user.department_name || '',
          color: user.department_color || 'bg-gray-500',
          description: '',
          isActive: true,
          userCount: 0,
          createdAt: new Date(),
          updatedAt: new Date()
        },
        role: user.role,
        permissions: user.permissions ? (typeof user.permissions === 'string' ? JSON.parse(user.permissions) : user.permissions) : [],
        isActive: Boolean(user.is_active),
        isOnline: Boolean(user.is_online),
        lastLoginAt: user.last_login_at ? new Date(user.last_login_at) : undefined,
        totalTickets: user.total_tickets || 0,
        totalRevenue: user.total_revenue || 0,
        createdAt: new Date(user.created_at),
        updatedAt: new Date(user.updated_at)
      }));

      setUsers(usersWithDates);
    } catch (error) {
      console.error('Error fetching department users:', error);
    }
  };

  const getRoleInfo = (role: string) => {
    const roleConfig = {
      super_admin: { color: 'bg-red-100 text-red-800', label: 'Sistem Yöneticisi', icon: Crown },
      admin: { color: 'bg-purple-100 text-purple-800', label: 'Acente Yöneticisi', icon: Shield },
      manager: { color: 'bg-blue-100 text-blue-800', label: 'Departman Yöneticisi', icon: Building2 },
      senior_agent: { color: 'bg-green-100 text-green-800', label: 'Kıdemli Acente', icon: UserCheck },
      agent: { color: 'bg-gray-100 text-gray-800', label: 'Acente', icon: Users },
      intern: { color: 'bg-yellow-100 text-yellow-800', label: 'Stajyer', icon: Clock }
    };

    return roleConfig[role as keyof typeof roleConfig] || roleConfig.agent;
  };

  const getPermissionLabel = (permission: string) => {
    for (const category of PERMISSION_CATEGORIES) {
      const perm = category.permissions.find(p => p.value === permission);
      if (perm) return perm.label;
    }
    return permission;
  };

  // Render kullanıcı bilgileri
  const renderUser = (user: AgencyUser) => (
    <div className="flex items-center">
      <div className="flex-shrink-0 h-10 w-10 rounded-full bg-gradient-to-r from-blue-600 to-purple-600 flex items-center justify-center">
        <span className="text-sm font-medium text-white">
          {user.firstName.charAt(0)}{user.lastName.charAt(0)}
        </span>
      </div>
      <div className="ml-4">
        <div className="text-sm font-medium text-gray-900 flex items-center">
          {user.firstName} {user.lastName}
          {user.isOnline && (
            <div className="ml-2 h-2 w-2 bg-green-500 rounded-full"></div>
          )}
          {user.id === department?.managerId && (
            <Crown className="ml-2 h-4 w-4 text-yellow-500" />
          )}
        </div>
        <div className="text-sm text-gray-500">{user.email}</div>
        <div className="text-xs text-gray-400">{user.position}</div>
      </div>
    </div>
  );

  // Render rol
  const renderRole = (role: string) => {
    const roleInfo = getRoleInfo(role);
    const IconComponent = roleInfo.icon;

    return (
      <span className={`inline-flex items-center px-2 py-1 text-xs font-semibold rounded-full ${roleInfo.color}`}>
        <IconComponent className="h-3 w-3 mr-1" />
        {roleInfo.label}
      </span>
    );
  };

  // Render performans
  const renderPerformance = (user: AgencyUser) => (
    <div>
      {user.totalTickets && (
        <div className="text-sm text-gray-900">{user.totalTickets} bilet</div>
      )}
      {user.totalRevenue && (
        <div className="text-sm text-green-600 font-medium">
          {user.totalRevenue.toLocaleString()} TRY
        </div>
      )}
    </div>
  );

  // Tablo sütunları
  const columns: TableColumn<AgencyUser>[] = [
    {
      key: 'user',
      title: 'Kullanıcı',
      render: (_, record) => renderUser(record)
    },
    {
      key: 'role',
      title: 'Rol',
      render: (_, record) => renderRole(record.role)
    },
    {
      key: 'permissions',
      title: 'Yetkiler',
      render: (_, record) => (
        <div className="text-sm text-gray-600">
          {record.permissions.length} yetki
        </div>
      )
    },
    {
      key: 'performance',
      title: 'Performans',
      render: (_, record) => renderPerformance(record)
    },
    {
      key: 'lastLogin',
      title: 'Son Giriş',
      render: (_, record) => (
        <div className="text-sm text-gray-500">
          {record.lastLoginAt?.toLocaleDateString('tr-TR')}
        </div>
      )
    }
  ];

  // Tablo aksiyonları
  const actions: TableAction<AgencyUser>[] = [
    {
      key: 'view',
      label: 'Görüntüle',
      onClick: (record) => window.location.href = `/users/${record.id}`,
      variant: 'primary'
    },
    {
      key: 'edit',
      label: 'Düzenle',
      onClick: (record) => window.location.href = `/users/${record.id}/edit`,
      variant: 'secondary'
    }
  ];

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse">
          <div className="h-8 bg-gray-200 rounded w-1/4 mb-6"></div>
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <div className="h-6 bg-gray-200 rounded w-1/3 mb-4"></div>
            <div className="space-y-3">
              <div className="h-4 bg-gray-200 rounded w-1/2"></div>
              <div className="h-4 bg-gray-200 rounded w-1/3"></div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (!department) {
    return (
      <div className="p-6">
        <div className="text-center py-12">
          <Building2 className="mx-auto h-12 w-12 text-gray-400" />
          <h3 className="mt-2 text-sm font-medium text-gray-900">Departman bulunamadı</h3>
          <p className="mt-1 text-sm text-gray-500">Bu departman mevcut değil veya silinmiş olabilir.</p>
          <div className="mt-6">
            <Link
              href="/departments"
              className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700"
            >
              <ArrowLeft className="mr-2 h-4 w-4" />
              Departmanlara Dön
            </Link>
          </div>
        </div>
      </div>
    );
  }

  // İstatistikler
  const stats = {
    totalUsers: users.length,
    activeUsers: users.filter(u => u.isActive).length,
    onlineUsers: users.filter(u => u.isOnline).length,
    totalRevenue: users.reduce((sum, u) => sum + (u.totalRevenue || 0), 0),
    totalTickets: users.reduce((sum, u) => sum + (u.totalTickets || 0), 0)
  };

  return (
    <div className="p-6">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-4">
          <Link
            href="/departments"
            className="flex items-center text-gray-600 hover:text-gray-900"
          >
            <ArrowLeft className="h-5 w-5 mr-2" />
            Departmanlara Dön
          </Link>
          <div className="h-6 border-l border-gray-300"></div>
          <div className="flex items-center">
            <div className={`w-4 h-4 rounded-full ${department.color} mr-3`}></div>
            <h1 className="text-2xl font-bold text-gray-900">{department.name}</h1>
          </div>
        </div>
        <div className="flex items-center space-x-3">
          <Link
            href={`/departments/${department.id}/edit`}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            <Edit className="h-4 w-4 mr-2" />
            Düzenle
          </Link>
        </div>
      </div>

      {/* Department Info Card */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <h2 className="text-lg font-semibold text-gray-900 mb-2">{department.name}</h2>
            <p className="text-gray-600 mb-4">{department.description}</p>
            
            <div className="space-y-2">
              <div className="flex items-center text-gray-600">
                <Calendar className="h-4 w-4 mr-2" />
                <span>Oluşturulma: {department.createdAt.toLocaleDateString('tr-TR')}</span>
              </div>
              <div className="flex items-center text-gray-600">
                <Users className="h-4 w-4 mr-2" />
                <span>{department.userCount} kullanıcı</span>
              </div>
            </div>
          </div>
          
          {department.manager && (
            <div>
              <h3 className="text-sm font-medium text-gray-500 mb-2">Departman Yöneticisi</h3>
              <div className="flex items-center">
                <div className="flex-shrink-0 h-12 w-12 rounded-full bg-gradient-to-r from-blue-600 to-purple-600 flex items-center justify-center">
                  <span className="text-sm font-medium text-white">
                    {department.manager.firstName.charAt(0)}{department.manager.lastName.charAt(0)}
                  </span>
                </div>
                <div className="ml-3">
                  <div className="text-sm font-medium text-gray-900 flex items-center">
                    {department.manager.firstName} {department.manager.lastName}
                    <Crown className="ml-2 h-4 w-4 text-yellow-500" />
                  </div>
                  <div className="text-sm text-gray-500">{department.manager.email}</div>
                  <div className="text-xs text-gray-400">{department.manager.position}</div>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-5 gap-4 mb-6">
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center">
            <Users className="h-8 w-8 text-blue-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam Kullanıcı</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.totalUsers}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center">
            <UserCheck className="h-8 w-8 text-green-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Aktif</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.activeUsers}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center">
            <Activity className="h-8 w-8 text-orange-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Çevrimiçi</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.onlineUsers}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center">
            <TrendingUp className="h-8 w-8 text-purple-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam Bilet</p>
              <p className="text-2xl font-semibold text-gray-900">{stats.totalTickets}</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-4 rounded-lg border border-gray-200">
          <div className="flex items-center">
            <TrendingUp className="h-8 w-8 text-green-600" />
            <div className="ml-3">
              <p className="text-sm font-medium text-gray-500">Toplam Gelir</p>
              <p className="text-xl font-semibold text-gray-900">{stats.totalRevenue.toLocaleString()} TRY</p>
            </div>
          </div>
        </div>
      </div>

      {/* Tabs */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200">
        <div className="border-b border-gray-200">
          <nav className="-mb-px flex space-x-8 px-6">
            <button
              onClick={() => setActiveTab('overview')}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'overview'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              Genel Bakış
            </button>
            <button
              onClick={() => setActiveTab('users')}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'users'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              <Users className="h-4 w-4 mr-2 inline" />
              Kullanıcılar ({users.length})
            </button>
            <button
              onClick={() => setActiveTab('settings')}
              className={`py-4 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'settings'
                  ? 'border-blue-500 text-blue-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              <Settings className="h-4 w-4 mr-2 inline" />
              Ayarlar
            </button>
          </nav>
        </div>

        <div className="p-6">
          {activeTab === 'overview' && (
            <div className="space-y-6">
              <div>
                <h3 className="text-lg font-medium text-gray-900 mb-4">Departman Hakkında</h3>
                <div className="bg-gray-50 rounded-lg p-4">
                  <p className="text-gray-700">{department.description}</p>
                </div>
              </div>

              {/* Departman Yetkileri */}
              <div>
                <h3 className="text-lg font-medium text-gray-900 mb-4 flex items-center">
                  <Shield className="h-5 w-5 mr-2" />
                  Departman Yetkileri
                </h3>
                {department.permissions && department.permissions.length > 0 ? (
                  <div className="bg-white rounded-lg border border-gray-200 p-4">
                    <div className="flex flex-wrap gap-2">
                      {department.permissions.map((perm) => (
                        <span
                          key={perm}
                          className="inline-flex items-center px-3 py-1.5 bg-blue-50 text-blue-700 rounded-lg text-sm font-medium border border-blue-200"
                        >
                          {getPermissionLabel(perm)}
                        </span>
                      ))}
                    </div>
                    <p className="mt-4 text-sm text-gray-500">
                      Toplam {department.permissions.length} yetki tanımlı. Bu departmandaki kullanıcılar otomatik olarak bu yetkileri alır.
                    </p>
                  </div>
                ) : (
                  <div className="bg-gray-50 rounded-lg p-4 border border-gray-200">
                    <p className="text-gray-500 text-sm">Bu departman için henüz yetki tanımlanmamış.</p>
                  </div>
                )}
              </div>
            </div>
          )}

          {activeTab === 'users' && (
            <div className="space-y-6">
              <div className="flex items-center justify-between">
                <h3 className="text-lg font-medium text-gray-900">Departman Kullanıcıları</h3>
                <Link
                  href={`/users/new?department=${department.id}`}
                  className="flex items-center px-3 py-2 text-sm bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  <UserPlus className="h-4 w-4 mr-2" />
                  Kullanıcı Ekle
                </Link>
              </div>
              
              <DataTable
                data={users}
                columns={columns}
                actions={actions}
                loading={false}
                emptyTitle="Kullanıcı bulunamadı"
                emptyDescription="Bu departmanda henüz kullanıcı bulunmuyor."
                emptyIcon={Users}
              />
            </div>
          )}

          {activeTab === 'settings' && (
            <div className="space-y-6">
              <h3 className="text-lg font-medium text-gray-900">Departman Ayarları</h3>
              
              <div className="bg-gray-50 rounded-lg p-4">
                <p className="text-gray-600">Departman ayarları yakında eklenecek...</p>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}